/*
 * Decompiled with CFR 0.152.
 */
package org.broadinstitute.sting.utils.nanoScheduler;

import com.google.java.contract.Ensures;
import com.google.java.contract.Requires;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Queue;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.LinkedBlockingDeque;
import org.apache.log4j.Logger;
import org.broadinstitute.sting.utils.AutoFormattingTime;
import org.broadinstitute.sting.utils.SimpleTimer;
import org.broadinstitute.sting.utils.collections.Pair;
import org.broadinstitute.sting.utils.exceptions.ReviewedStingException;
import org.broadinstitute.sting.utils.nanoScheduler.NanoSchedulerMapFunction;
import org.broadinstitute.sting.utils.nanoScheduler.NanoSchedulerProgressFunction;
import org.broadinstitute.sting.utils.nanoScheduler.NanoSchedulerReduceFunction;

public class NanoScheduler<InputType, MapType, ReduceType> {
    private static final Logger logger = Logger.getLogger(NanoScheduler.class);
    private static final boolean ALLOW_SINGLE_THREAD_FASTPATH = true;
    private static final boolean LOG_MAP_TIMES = false;
    private static final boolean TIME_CALLS = true;
    final int bufferSize;
    final int nThreads;
    final ExecutorService inputExecutor;
    final ExecutorService mapExecutor;
    boolean shutdown = false;
    boolean debug = false;
    private NanoSchedulerProgressFunction<InputType> progressFunction = null;
    final SimpleTimer outsideSchedulerTimer = new SimpleTimer("outside");
    final SimpleTimer inputTimer = new SimpleTimer("input");
    final SimpleTimer mapTimer = new SimpleTimer("map");
    final SimpleTimer reduceTimer = new SimpleTimer("reduce");

    public NanoScheduler(int bufferSize, int nThreads) {
        if (bufferSize < 1) {
            throw new IllegalArgumentException("bufferSize must be >= 1, got " + bufferSize);
        }
        if (nThreads < 1) {
            throw new IllegalArgumentException("nThreads must be >= 1, got " + nThreads);
        }
        this.bufferSize = bufferSize;
        this.nThreads = nThreads;
        this.mapExecutor = nThreads == 1 ? null : Executors.newFixedThreadPool(nThreads - 1);
        this.inputExecutor = Executors.newSingleThreadExecutor();
        this.outsideSchedulerTimer.start();
    }

    @Ensures(value={"result > 0"})
    public int getnThreads() {
        return this.nThreads;
    }

    @Ensures(value={"result > 0"})
    public int getBufferSize() {
        return this.bufferSize;
    }

    public void shutdown() {
        List<Runnable> remaining;
        this.outsideSchedulerTimer.stop();
        if (this.mapExecutor != null && !(remaining = this.mapExecutor.shutdownNow()).isEmpty()) {
            throw new IllegalStateException("Remaining tasks found in the mapExecutor, unexpected behavior!");
        }
        this.shutdown = true;
        this.printTimerInfo("Input   time", this.inputTimer);
        this.printTimerInfo("Map     time", this.mapTimer);
        this.printTimerInfo("Reduce  time", this.reduceTimer);
        this.printTimerInfo("Outside time", this.outsideSchedulerTimer);
    }

    private void printTimerInfo(String label, SimpleTimer timer) {
        double total = this.inputTimer.getElapsedTime() + this.mapTimer.getElapsedTime() + this.reduceTimer.getElapsedTime() + this.outsideSchedulerTimer.getElapsedTime();
        double myTimeInSec = timer.getElapsedTime();
        double myTimePercent = myTimeInSec / total * 100.0;
        logger.info(String.format("%s: %s (%5.2f%%)", label, new AutoFormattingTime(myTimeInSec), myTimePercent));
    }

    public boolean isShutdown() {
        return this.shutdown;
    }

    public boolean isDebug() {
        return this.debug;
    }

    private void debugPrint(String format, Object ... args) {
        if (this.isDebug()) {
            logger.info("Thread " + Thread.currentThread().getId() + ":" + String.format(format, args));
        }
    }

    public void setDebug(boolean debug) {
        this.debug = debug;
    }

    public void setProgressFunction(NanoSchedulerProgressFunction<InputType> progressFunction) {
        this.progressFunction = progressFunction;
    }

    public ReduceType execute(Iterator<InputType> inputReader, NanoSchedulerMapFunction<InputType, MapType> map, ReduceType initialValue, NanoSchedulerReduceFunction<MapType, ReduceType> reduce) {
        if (this.isShutdown()) {
            throw new IllegalStateException("execute called on already shutdown NanoScheduler");
        }
        if (inputReader == null) {
            throw new IllegalArgumentException("inputReader cannot be null");
        }
        if (map == null) {
            throw new IllegalArgumentException("map function cannot be null");
        }
        if (reduce == null) {
            throw new IllegalArgumentException("reduce function cannot be null");
        }
        this.outsideSchedulerTimer.stop();
        ReduceType result = this.getnThreads() == 1 ? this.executeSingleThreaded(inputReader, map, initialValue, reduce) : this.executeMultiThreaded(inputReader, map, initialValue, reduce);
        this.outsideSchedulerTimer.restart();
        return result;
    }

    private ReduceType executeSingleThreaded(Iterator<InputType> inputReader, NanoSchedulerMapFunction<InputType, MapType> map, ReduceType initialValue, NanoSchedulerReduceFunction<MapType, ReduceType> reduce) {
        ReduceType sum = initialValue;
        int i = 0;
        this.inputTimer.restart();
        while (inputReader.hasNext()) {
            InputType input = inputReader.next();
            this.inputTimer.stop();
            this.mapTimer.restart();
            long preMapTime = this.mapTimer.currentTimeNano();
            MapType mapValue = map.apply(input);
            this.mapTimer.stop();
            if (i++ % this.bufferSize == 0 && this.progressFunction != null) {
                this.progressFunction.progress(input);
            }
            this.reduceTimer.restart();
            sum = reduce.apply(mapValue, sum);
            this.reduceTimer.stop();
            this.inputTimer.restart();
        }
        return sum;
    }

    private ReduceType executeMultiThreaded(Iterator<InputType> inputReader, NanoSchedulerMapFunction<InputType, MapType> map, ReduceType initialValue, NanoSchedulerReduceFunction<MapType, ReduceType> reduce) {
        this.debugPrint("Executing nanoScheduler", new Object[0]);
        ReduceType sum = initialValue;
        boolean done = false;
        LinkedBlockingDeque<InputDatum> inputQueue = new LinkedBlockingDeque<InputDatum>(this.bufferSize);
        this.inputExecutor.submit(new InputProducer(inputReader, inputQueue));
        while (!done) {
            try {
                Pair<List<InputType>, Boolean> readResults = this.readInputs(inputQueue);
                List<InputType> inputs = readResults.getFirst();
                done = readResults.getSecond();
                if (!inputs.isEmpty()) {
                    Queue<Future<MapType>> mapQueue = this.submitMapJobs(map, this.mapExecutor, inputs);
                    sum = this.reduceSerial(reduce, mapQueue, sum);
                    this.debugPrint("  Done with cycle of map/reduce", new Object[0]);
                    if (this.progressFunction == null) continue;
                    this.progressFunction.progress(inputs.get(inputs.size() - 1));
                    continue;
                }
                if (done) continue;
                throw new IllegalStateException("Inputs empty but not done");
            }
            catch (InterruptedException ex) {
                throw new ReviewedStingException("got execution exception", ex);
            }
            catch (ExecutionException ex) {
                throw new ReviewedStingException("got execution exception", ex);
            }
        }
        return sum;
    }

    @Requires(value={"reduce != null", "! mapQueue.isEmpty()"})
    private ReduceType reduceSerial(NanoSchedulerReduceFunction<MapType, ReduceType> reduce, Queue<Future<MapType>> mapQueue, ReduceType initSum) throws InterruptedException, ExecutionException {
        ReduceType sum = initSum;
        for (Future future : mapQueue) {
            Object value = future.get();
            this.reduceTimer.restart();
            sum = reduce.apply(value, sum);
            this.reduceTimer.stop();
        }
        return sum;
    }

    @Requires(value={"inputReader != null"})
    @Ensures(value={"result != null"})
    private Pair<List<InputType>, Boolean> readInputs(BlockingQueue<InputDatum> inputReader) throws InterruptedException {
        int n = 0;
        LinkedList inputs = new LinkedList();
        boolean done = false;
        while (!done && n < this.getBufferSize()) {
            InputDatum input = inputReader.take();
            done = input.isLast();
            if (done) continue;
            inputs.add(input.datum);
            ++n;
        }
        return new Pair<List<InputType>, Boolean>(inputs, done);
    }

    @Requires(value={"map != null", "! inputs.isEmpty()"})
    private Queue<Future<MapType>> submitMapJobs(NanoSchedulerMapFunction<InputType, MapType> map, ExecutorService executor, List<InputType> inputs) {
        LinkedList<Future<MapType>> mapQueue = new LinkedList<Future<MapType>>();
        for (InputType input : inputs) {
            CallableMap doMap = new CallableMap(map, input);
            Future future = executor.submit(doMap);
            mapQueue.add(future);
        }
        return mapQueue;
    }

    private class CallableMap
    implements Callable<MapType> {
        final InputType input;
        final NanoSchedulerMapFunction<InputType, MapType> map;

        @Requires(value={"map != null"})
        private CallableMap(NanoSchedulerMapFunction<InputType, MapType> map, InputType inputs) {
            this.input = inputs;
            this.map = map;
        }

        @Override
        public MapType call() throws Exception {
            NanoScheduler.this.mapTimer.restart();
            if (NanoScheduler.this.debug) {
                NanoScheduler.this.debugPrint("\t\tmap " + this.input, new Object[0]);
            }
            Object result = this.map.apply(this.input);
            NanoScheduler.this.mapTimer.stop();
            return result;
        }
    }

    private class InputDatum {
        final boolean isLast;
        final InputType datum;

        private InputDatum(InputType datum) {
            this.isLast = false;
            this.datum = datum;
        }

        private InputDatum() {
            this.isLast = true;
            this.datum = null;
        }

        public boolean isLast() {
            return this.isLast;
        }
    }

    private class InputProducer
    implements Runnable {
        final Iterator<InputType> inputReader;
        final BlockingQueue<InputDatum> outputQueue;

        public InputProducer(Iterator<InputType> inputReader, BlockingQueue<InputDatum> outputQueue) {
            this.inputReader = inputReader;
            this.outputQueue = outputQueue;
        }

        @Override
        public void run() {
            try {
                while (this.inputReader.hasNext()) {
                    NanoScheduler.this.inputTimer.restart();
                    Object input = this.inputReader.next();
                    NanoScheduler.this.inputTimer.stop();
                    this.outputQueue.put(new InputDatum(input));
                }
                this.outputQueue.put(new InputDatum());
            }
            catch (InterruptedException ex) {
                throw new ReviewedStingException("got execution exception", ex);
            }
        }
    }
}

