/*
 * Decompiled with CFR 0.152.
 */
package org.broadinstitute.sting.gatk.datasources.reads;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.channels.FileChannel;
import java.util.ArrayList;
import java.util.Arrays;
import net.sf.samtools.Bin;
import net.sf.samtools.GATKBin;
import net.sf.samtools.GATKChunk;
import net.sf.samtools.LinearIndex;
import org.broadinstitute.sting.gatk.datasources.reads.GATKBAMIndexData;
import org.broadinstitute.sting.utils.exceptions.ReviewedStingException;
import org.broadinstitute.sting.utils.exceptions.UserException;

public class GATKBAMIndex {
    private static final byte[] BAM_INDEX_MAGIC = "BAI\u0001".getBytes();
    protected static final int BIN_GENOMIC_SPAN = 0x20000000;
    private static final int[] LEVEL_STARTS = new int[]{0, 1, 9, 73, 585, 4681};
    public static final int MAX_BINS = 37450;
    private final File mFile;
    private final int sequenceCount;
    private final long[] sequenceStartCache;
    private FileInputStream fileStream;
    private FileChannel fileChannel;
    private static final int INT_SIZE_IN_BYTES = 4;
    private static final int LONG_SIZE_IN_BYTES = 8;
    private ByteBuffer buffer = null;

    public GATKBAMIndex(File file) {
        this.mFile = file;
        this.openIndexFile();
        this.seek(0L);
        byte[] buffer = this.readBytes(4);
        if (!Arrays.equals(buffer, BAM_INDEX_MAGIC)) {
            throw new ReviewedStingException("Invalid file header in BAM index " + this.mFile + ": " + new String(buffer));
        }
        this.seek(4L);
        this.sequenceCount = this.readInteger();
        this.sequenceStartCache = new long[this.sequenceCount];
        for (int i = 1; i < this.sequenceCount; ++i) {
            this.sequenceStartCache[i] = -1L;
        }
        if (this.sequenceCount > 0) {
            this.sequenceStartCache[0] = this.position();
        }
        this.closeIndexFile();
    }

    public GATKBAMIndexData readReferenceSequence(int referenceSequence) {
        this.openIndexFile();
        if (referenceSequence >= this.sequenceCount) {
            throw new ReviewedStingException("Invalid sequence number " + referenceSequence + " in index file " + this.mFile);
        }
        this.skipToSequence(referenceSequence);
        int binCount = this.readInteger();
        ArrayList<GATKBin> bins = new ArrayList<GATKBin>();
        for (int binNumber = 0; binNumber < binCount; ++binNumber) {
            int indexBin = this.readInteger();
            int nChunks = this.readInteger();
            ArrayList<GATKChunk> chunks = new ArrayList<GATKChunk>(nChunks);
            long[] rawChunkData = this.readLongs(nChunks * 2);
            for (int ci = 0; ci < nChunks; ++ci) {
                long chunkBegin = rawChunkData[ci * 2];
                long chunkEnd = rawChunkData[ci * 2 + 1];
                chunks.add(new GATKChunk(chunkBegin, chunkEnd));
            }
            GATKBin bin = new GATKBin(referenceSequence, indexBin);
            bin.setChunkList(chunks.toArray(new GATKChunk[chunks.size()]));
            while (indexBin >= bins.size()) {
                bins.add(null);
            }
            bins.set(indexBin, bin);
        }
        int nLinearBins = this.readInteger();
        long[] linearIndexEntries = this.readLongs(nLinearBins);
        LinearIndex linearIndex = new LinearIndex(referenceSequence, 0, linearIndexEntries);
        this.closeIndexFile();
        return new GATKBAMIndexData(this, referenceSequence, bins, linearIndex);
    }

    public static int getNumIndexLevels() {
        return LEVEL_STARTS.length;
    }

    public static int getFirstBinInLevel(int levelNumber) {
        return LEVEL_STARTS[levelNumber];
    }

    public int getLevelSize(int levelNumber) {
        if (levelNumber == GATKBAMIndex.getNumIndexLevels() - 1) {
            return 37450 - LEVEL_STARTS[levelNumber] - 1;
        }
        return LEVEL_STARTS[levelNumber + 1] - LEVEL_STARTS[levelNumber];
    }

    public int getLevelForBin(Bin bin) {
        GATKBin gatkBin = new GATKBin(bin);
        if (gatkBin.getBinNumber() >= 37450) {
            throw new ReviewedStingException("Tried to get level for invalid bin in index file " + this.mFile);
        }
        for (int i = GATKBAMIndex.getNumIndexLevels() - 1; i >= 0; --i) {
            if (gatkBin.getBinNumber() < LEVEL_STARTS[i]) continue;
            return i;
        }
        throw new ReviewedStingException("Unable to find correct bin for bin " + bin + " in index file " + this.mFile);
    }

    public int getFirstLocusInBin(Bin bin) {
        int level = this.getLevelForBin(bin);
        int levelStart = LEVEL_STARTS[level];
        int levelSize = (level == GATKBAMIndex.getNumIndexLevels() - 1 ? 37449 : LEVEL_STARTS[level + 1]) - levelStart;
        return (new GATKBin(bin).getBinNumber() - levelStart) * (0x20000000 / levelSize) + 1;
    }

    public int getLastLocusInBin(Bin bin) {
        int level = this.getLevelForBin(bin);
        int levelStart = LEVEL_STARTS[level];
        int levelSize = (level == GATKBAMIndex.getNumIndexLevels() - 1 ? 37449 : LEVEL_STARTS[level + 1]) - levelStart;
        return (new GATKBin(bin).getBinNumber() - levelStart + 1) * (0x20000000 / levelSize);
    }

    public long getStartOfLastLinearBin() {
        this.openIndexFile();
        this.seek(4L);
        int sequenceCount = this.readInteger();
        long lastLinearIndexPointer = -1L;
        for (int i = 0; i < sequenceCount; ++i) {
            int nBins = this.readInteger();
            for (int j1 = 0; j1 < nBins; ++j1) {
                this.skipBytes(4);
                int nChunks = this.readInteger();
                this.skipBytes(16 * nChunks);
            }
            int nLinearBins = this.readInteger();
            if (nLinearBins <= 0) continue;
            this.skipBytes(8 * (nLinearBins - 1));
            lastLinearIndexPointer = this.readLongs(1)[0];
        }
        this.closeIndexFile();
        return lastLinearIndexPointer;
    }

    protected int getMaxAddressibleGenomicLocation() {
        return 0x20000000;
    }

    protected void skipToSequence(int referenceSequence) {
        int sequenceIndex = referenceSequence;
        while (this.sequenceStartCache[sequenceIndex] == -1L) {
            --sequenceIndex;
        }
        this.seek(this.sequenceStartCache[sequenceIndex]);
        for (int i = sequenceIndex; i < referenceSequence; ++i) {
            this.sequenceStartCache[i] = this.position();
            int nBins = this.readInteger();
            for (int j = 0; j < nBins; ++j) {
                int bin = this.readInteger();
                int nChunks = this.readInteger();
                this.skipBytes(16 * nChunks);
            }
            int nLinearBins = this.readInteger();
            this.skipBytes(8 * nLinearBins);
        }
        this.sequenceStartCache[referenceSequence] = this.position();
    }

    private void openIndexFile() {
        try {
            this.fileStream = new FileInputStream(this.mFile);
            this.fileChannel = this.fileStream.getChannel();
        }
        catch (IOException exc) {
            throw new ReviewedStingException("Unable to open index file (" + exc.getMessage() + ")" + this.mFile, exc);
        }
    }

    private void closeIndexFile() {
        try {
            this.fileChannel.close();
            this.fileStream.close();
        }
        catch (IOException exc) {
            throw new ReviewedStingException("Unable to close index file " + this.mFile, exc);
        }
    }

    private byte[] readBytes(int count) {
        ByteBuffer buffer = this.getBuffer(count);
        this.read(buffer);
        buffer.flip();
        byte[] contents = new byte[count];
        buffer.get(contents);
        return contents;
    }

    private int readInteger() {
        ByteBuffer buffer = this.getBuffer(4);
        this.read(buffer);
        buffer.flip();
        return buffer.getInt();
    }

    private long[] readLongs(int count) {
        ByteBuffer buffer = this.getBuffer(count * 8);
        this.read(buffer);
        buffer.flip();
        long[] result = new long[count];
        for (int i = 0; i < count; ++i) {
            result[i] = buffer.getLong();
        }
        return result;
    }

    private void read(ByteBuffer buffer) {
        try {
            int bytesExpected = buffer.limit();
            int bytesRead = this.fileChannel.read(buffer);
            if (bytesRead < bytesExpected) {
                throw new UserException.MalformedFile(this.mFile, String.format("Premature end-of-file while reading BAM index file %s. It's likely that this file is truncated or corrupt -- Please try re-indexing the corresponding BAM file.", this.mFile));
            }
        }
        catch (IOException ex) {
            throw new ReviewedStingException("Index: unable to read bytes from index file " + this.mFile);
        }
    }

    private ByteBuffer getBuffer(int size) {
        if (this.buffer == null || this.buffer.capacity() < size) {
            this.buffer = ByteBuffer.allocate(size);
            this.buffer.order(ByteOrder.LITTLE_ENDIAN);
        }
        this.buffer.clear();
        this.buffer.limit(size);
        return this.buffer;
    }

    private void skipBytes(int count) {
        try {
            this.fileChannel.position(this.fileChannel.position() + (long)count);
        }
        catch (IOException ex) {
            throw new ReviewedStingException("Index: unable to reposition file channel of index file " + this.mFile);
        }
    }

    private void seek(long position) {
        try {
            this.fileChannel.position(position);
        }
        catch (IOException ex) {
            throw new ReviewedStingException("Index: unable to reposition of file channel of index file " + this.mFile);
        }
    }

    private long position() {
        try {
            return this.fileChannel.position();
        }
        catch (IOException exc) {
            throw new ReviewedStingException("Unable to read position from index file " + this.mFile, exc);
        }
    }
}

