/*
 * Decompiled with CFR 0.152.
 */
package org.broadinstitute.sting.gatk.datasources.reference;

import java.io.File;
import java.util.ArrayList;
import java.util.Collections;
import net.sf.picard.reference.FastaSequenceIndex;
import net.sf.picard.reference.FastaSequenceIndexBuilder;
import net.sf.picard.reference.IndexedFastaSequenceFile;
import net.sf.picard.sam.CreateSequenceDictionary;
import net.sf.samtools.SAMSequenceRecord;
import org.apache.log4j.Logger;
import org.broadinstitute.sting.gatk.datasources.reads.LocusShard;
import org.broadinstitute.sting.gatk.datasources.reads.SAMDataSource;
import org.broadinstitute.sting.gatk.datasources.reads.Shard;
import org.broadinstitute.sting.utils.GenomeLoc;
import org.broadinstitute.sting.utils.GenomeLocParser;
import org.broadinstitute.sting.utils.GenomeLocSortedSet;
import org.broadinstitute.sting.utils.exceptions.ReviewedStingException;
import org.broadinstitute.sting.utils.exceptions.UserException;
import org.broadinstitute.sting.utils.fasta.CachingIndexedFastaSequenceFile;
import org.broadinstitute.sting.utils.file.FSLockWithShared;
import org.broadinstitute.sting.utils.file.FileSystemInabilityToLockException;

public class ReferenceDataSource {
    private IndexedFastaSequenceFile reference;
    protected static final Logger logger = Logger.getLogger(ReferenceDataSource.class);

    public ReferenceDataSource(File fastaFile) {
        FSLockWithShared dictLock;
        if (!fastaFile.exists()) {
            throw new UserException("The fasta file you specified (" + fastaFile.getAbsolutePath() + ") does not exist.");
        }
        boolean isGzipped = fastaFile.getAbsolutePath().endsWith(".gz");
        File indexFile = new File(fastaFile.getAbsolutePath() + ".fai");
        String fastaExt = (fastaFile.getAbsolutePath().endsWith("fa") ? ".fa" : ".fasta") + (isGzipped ? ".gz" : "");
        File dictFile = new File(fastaFile.getAbsolutePath().replace(fastaExt, ".dict"));
        if (!indexFile.exists()) {
            if (isGzipped) {
                throw new UserException.CouldNotCreateReferenceFAIorDictForGzippedRef(fastaFile);
            }
            logger.info(String.format("Index file %s does not exist. Trying to create it now.", indexFile.getAbsolutePath()));
            FSLockWithShared indexLock = new FSLockWithShared(indexFile, true);
            try {
                if (!indexLock.exclusiveLock()) {
                    throw new UserException.CouldNotCreateReferenceIndexFileBecauseOfLock(dictFile);
                }
                FastaSequenceIndexBuilder faiBuilder = new FastaSequenceIndexBuilder(fastaFile, true);
                FastaSequenceIndex sequenceIndex = faiBuilder.createIndex();
                FastaSequenceIndexBuilder.saveAsFaiFile(sequenceIndex, indexFile);
            }
            catch (FileSystemInabilityToLockException ex) {
                logger.info("Unable to create write lock: " + ex.getMessage());
                logger.info("Skipping index creation.");
            }
            catch (UserException e) {
                throw e;
            }
            catch (Exception e) {
                throw new UserException.CouldNotCreateReferenceIndexFile(indexFile, e);
            }
            finally {
                indexLock.unlock();
            }
        }
        if (!dictFile.exists()) {
            if (isGzipped) {
                throw new UserException.CouldNotCreateReferenceFAIorDictForGzippedRef(fastaFile);
            }
            logger.info(String.format("Dict file %s does not exist. Trying to create it now.", dictFile.getAbsolutePath()));
            dictLock = new FSLockWithShared(dictFile, true);
            try {
                if (!dictLock.exclusiveLock()) {
                    throw new UserException.CouldNotCreateReferenceIndexFileBecauseOfLock(dictFile);
                }
                File tempFile = File.createTempFile("dict", null, dictFile.getParentFile());
                tempFile.deleteOnExit();
                String[] args = new String[]{String.format("r=%s", fastaFile.getAbsolutePath()), String.format("o=%s", tempFile.getAbsolutePath())};
                new CreateSequenceDictionary().instanceMain(args);
                if (!tempFile.renameTo(dictFile)) {
                    throw new UserException("Error transferring temp file " + tempFile + " to dict file " + dictFile);
                }
            }
            catch (FileSystemInabilityToLockException ex) {
                logger.info("Unable to create write lock: " + ex.getMessage());
                logger.info("Skipping dictionary creation.");
            }
            catch (Exception e) {
                throw new UserException.CouldNotCreateReferenceIndexFile(dictFile, e);
            }
            finally {
                dictLock.unlock();
            }
        }
        dictLock = new FSLockWithShared(dictFile, true);
        FSLockWithShared indexLock = new FSLockWithShared(indexFile, true);
        try {
            try {
                if (!dictLock.sharedLock()) {
                    throw new ReviewedStingException("Could not open dictionary file because a lock could not be obtained.");
                }
            }
            catch (FileSystemInabilityToLockException ex) {
                logger.info(String.format("Unable to create a lock on dictionary file: %s", ex.getMessage()));
                logger.info("Treating existing dictionary file as complete.");
            }
            try {
                if (!indexLock.sharedLock()) {
                    throw new ReviewedStingException("Could not open index file because a lock could not be obtained.");
                }
            }
            catch (FileSystemInabilityToLockException ex) {
                logger.info(String.format("Unable to create a lock on index file: %s", ex.getMessage()));
                logger.info("Treating existing index file as complete.");
            }
            this.reference = new CachingIndexedFastaSequenceFile(fastaFile);
        }
        catch (IllegalArgumentException e) {
            throw new UserException.CouldNotReadInputFile(fastaFile, "Could not read reference sequence.  The FASTA must have either a .fasta or .fa extension", e);
        }
        catch (Exception e) {
            throw new UserException.CouldNotReadInputFile(fastaFile, e);
        }
        finally {
            dictLock.unlock();
            indexLock.unlock();
        }
    }

    public IndexedFastaSequenceFile getReference() {
        return this.reference;
    }

    public Iterable<Shard> createShardsOverEntireReference(SAMDataSource readsDataSource, GenomeLocParser parser, int maxShardSize) {
        ArrayList<Shard> shards = new ArrayList<Shard>();
        for (SAMSequenceRecord refSequenceRecord : this.reference.getSequenceDictionary().getSequences()) {
            for (int shardStart = 1; shardStart <= refSequenceRecord.getSequenceLength(); shardStart += maxShardSize) {
                int shardStop = Math.min(shardStart + maxShardSize - 1, refSequenceRecord.getSequenceLength());
                shards.add(new LocusShard(parser, readsDataSource, Collections.singletonList(parser.createGenomeLoc(refSequenceRecord.getSequenceName(), shardStart, shardStop)), null));
            }
        }
        return shards;
    }

    public Iterable<Shard> createShardsOverIntervals(SAMDataSource readsDataSource, GenomeLocSortedSet intervals, int maxShardSize) {
        ArrayList<Shard> shards = new ArrayList<Shard>();
        for (GenomeLoc interval : intervals) {
            while (interval.size() > maxShardSize) {
                shards.add(new LocusShard(intervals.getGenomeLocParser(), readsDataSource, Collections.singletonList(intervals.getGenomeLocParser().createGenomeLoc(interval.getContig(), interval.getStart(), interval.getStart() + maxShardSize - 1)), null));
                interval = intervals.getGenomeLocParser().createGenomeLoc(interval.getContig(), interval.getStart() + maxShardSize, interval.getStop());
            }
            shards.add(new LocusShard(intervals.getGenomeLocParser(), readsDataSource, Collections.singletonList(interval), null));
        }
        return shards;
    }
}

