/*
 * Decompiled with CFR 0.152.
 */
package org.broadinstitute.sting.utils.threading;

import com.google.java.contract.Ensures;
import com.google.java.contract.Invariant;
import com.google.java.contract.Requires;
import java.lang.management.ManagementFactory;
import java.lang.management.ThreadInfo;
import java.lang.management.ThreadMXBean;
import java.util.EnumMap;
import java.util.concurrent.TimeUnit;
import org.apache.log4j.Logger;
import org.apache.log4j.Priority;
import org.broadinstitute.sting.utils.AutoFormattingTime;
import org.broadinstitute.sting.utils.threading.EfficiencyMonitoringThreadFactory;

@Invariant(value={"nThreadsAnalyzed >= 0"})
public class ThreadEfficiencyMonitor {
    protected static final boolean DEBUG = false;
    protected static Logger logger = Logger.getLogger(EfficiencyMonitoringThreadFactory.class);
    final EnumMap<State, Long> times = new EnumMap(State.class);
    int nThreadsAnalyzed = 0;
    final ThreadMXBean bean = ManagementFactory.getThreadMXBean();

    public ThreadEfficiencyMonitor() {
        if (this.bean.isThreadContentionMonitoringSupported()) {
            this.bean.setThreadContentionMonitoringEnabled(true);
        } else {
            logger.warn("Thread contention monitoring not supported, we cannot track GATK multi-threaded efficiency");
        }
        if (this.bean.isThreadCpuTimeSupported()) {
            this.bean.setThreadCpuTimeEnabled(true);
        } else {
            logger.warn("Thread CPU monitoring not supported, we cannot track GATK multi-threaded efficiency");
        }
        for (State state : State.values()) {
            this.times.put(state, 0L);
        }
    }

    private static long nanoToMilli(long timeInNano) {
        return TimeUnit.NANOSECONDS.toMillis(timeInNano);
    }

    @Ensures(value={"result >= 0"})
    public synchronized long getStateTime(State state) {
        return this.times.get((Object)state);
    }

    @Ensures(value={"result >= 0"})
    public synchronized long getTotalTime() {
        long total = 0L;
        for (long time : this.times.values()) {
            total += time;
        }
        return total;
    }

    @Ensures(value={"result >= 0.0", "result <= 100.0"})
    public synchronized double getStatePercent(State state) {
        return 100.0 * (double)this.getStateTime(state) / (double)Math.max(this.getTotalTime(), 1L);
    }

    public int getnThreadsAnalyzed() {
        return this.nThreadsAnalyzed;
    }

    public synchronized String toString() {
        StringBuilder b = new StringBuilder();
        b.append("total ").append(this.getTotalTime()).append(" ");
        for (State state : State.values()) {
            b.append((Object)state).append(" ").append(this.getStateTime(state)).append(" ");
        }
        return b.toString();
    }

    public synchronized void printUsageInformation(Logger logger) {
        this.printUsageInformation(logger, Priority.INFO);
    }

    public synchronized void printUsageInformation(Logger logger, Priority priority) {
        logger.debug("Number of threads monitored: " + this.getnThreadsAnalyzed());
        logger.debug("Total runtime " + new AutoFormattingTime(TimeUnit.MILLISECONDS.toSeconds(this.getTotalTime())));
        for (State state : State.values()) {
            logger.debug(String.format("\tPercent of time spent %s is %.2f", state.getUserFriendlyName(), this.getStatePercent(state)));
        }
        logger.log(priority, String.format("CPU      efficiency : %6.2f%% of time spent %s", this.getStatePercent(State.USER_CPU), State.USER_CPU.getUserFriendlyName()));
        logger.log(priority, String.format("Walker inefficiency : %6.2f%% of time spent %s", this.getStatePercent(State.BLOCKING), State.BLOCKING.getUserFriendlyName()));
        logger.log(priority, String.format("I/O    inefficiency : %6.2f%% of time spent %s", this.getStatePercent(State.WAITING_FOR_IO), State.WAITING_FOR_IO.getUserFriendlyName()));
        logger.log(priority, String.format("Thread inefficiency : %6.2f%% of time spent %s", this.getStatePercent(State.WAITING), State.WAITING.getUserFriendlyName()));
    }

    @Ensures(value={"getTotalTime() >= old(getTotalTime())"})
    public synchronized void threadIsDone(Thread thread) {
        ++this.nThreadsAnalyzed;
        long threadID = thread.getId();
        ThreadInfo info = this.bean.getThreadInfo(thread.getId());
        long totalTimeNano = this.bean.getThreadCpuTime(threadID);
        long userTimeNano = this.bean.getThreadUserTime(threadID);
        long systemTimeNano = totalTimeNano - userTimeNano;
        long userTimeInMilliseconds = ThreadEfficiencyMonitor.nanoToMilli(userTimeNano);
        long systemTimeInMilliseconds = ThreadEfficiencyMonitor.nanoToMilli(systemTimeNano);
        if (info != null) {
            this.incTimes(State.BLOCKING, info.getBlockedTime());
            this.incTimes(State.WAITING, info.getWaitedTime());
            this.incTimes(State.USER_CPU, userTimeInMilliseconds);
            this.incTimes(State.WAITING_FOR_IO, systemTimeInMilliseconds);
        }
    }

    @Requires(value={"state != null", "by >= 0"})
    @Ensures(value={"getTotalTime() == old(getTotalTime()) + by"})
    private synchronized void incTimes(State state, long by) {
        this.times.put(state, this.times.get((Object)state) + by);
    }

    public static enum State {
        BLOCKING("blocking on synchronized data structures"),
        WAITING("waiting on some other thread"),
        USER_CPU("doing productive CPU work"),
        WAITING_FOR_IO("waiting for I/O");

        private final String userFriendlyName;

        private State(String userFriendlyName) {
            this.userFriendlyName = userFriendlyName;
        }

        public String getUserFriendlyName() {
            return this.userFriendlyName;
        }
    }
}

