/*
 * Decompiled with CFR 0.152.
 */
package org.broadinstitute.sting.utils.codecs.bcf2;

import com.google.java.contract.Ensures;
import com.google.java.contract.Requires;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Arrays;
import org.apache.log4j.Logger;
import org.broad.tribble.FeatureCodec;
import org.broadinstitute.sting.utils.codecs.bcf2.BCF2Type;
import org.broadinstitute.sting.utils.codecs.bcf2.BCF2Utils;
import org.broadinstitute.sting.utils.exceptions.ReviewedStingException;
import org.broadinstitute.sting.utils.exceptions.UserException;

public final class BCF2Decoder {
    protected static final Logger logger = Logger.getLogger(FeatureCodec.class);
    byte[] recordBytes = null;
    ByteArrayInputStream recordStream = null;

    public BCF2Decoder() {
    }

    protected BCF2Decoder(byte[] recordBytes) {
        this.setRecordBytes(recordBytes);
    }

    public void readNextBlock(int blockSizeInBytes, InputStream stream) {
        if (blockSizeInBytes < 0) {
            throw new UserException.MalformedBCF2("Invalid block size " + blockSizeInBytes);
        }
        this.setRecordBytes(BCF2Decoder.readRecordBytes(blockSizeInBytes, stream));
    }

    public void skipNextBlock(int blockSizeInBytes, InputStream stream) {
        try {
            int bytesRead = (int)stream.skip(blockSizeInBytes);
            BCF2Decoder.validateReadBytes(bytesRead, 1, blockSizeInBytes);
        }
        catch (IOException e) {
            throw new UserException.CouldNotReadInputFile("I/O error while reading BCF2 file", (Exception)e);
        }
        this.recordBytes = null;
        this.recordStream = null;
    }

    public byte[] getRecordBytes() {
        return this.recordBytes;
    }

    public int getBlockSize() {
        return this.recordBytes.length;
    }

    public boolean blockIsFullyDecoded() {
        return this.recordStream.available() == 0;
    }

    @Requires(value={"recordBytes != null"})
    @Ensures(value={"this.recordBytes == recordBytes", "recordStream != null"})
    public void setRecordBytes(byte[] recordBytes) {
        this.recordBytes = recordBytes;
        this.recordStream = new ByteArrayInputStream(recordBytes);
    }

    public final Object decodeTypedValue() throws IOException {
        byte typeDescriptor = this.readTypeDescriptor();
        return this.decodeTypedValue(typeDescriptor);
    }

    public final Object decodeTypedValue(byte typeDescriptor) throws IOException {
        int size = this.decodeNumberOfElements(typeDescriptor);
        return this.decodeTypedValue(typeDescriptor, size);
    }

    @Requires(value={"size >= 0"})
    public final Object decodeTypedValue(byte typeDescriptor, int size) throws IOException {
        if (size == 0) {
            return null;
        }
        BCF2Type type = BCF2Utils.decodeType(typeDescriptor);
        if (type == BCF2Type.CHAR) {
            return this.decodeLiteralString(size);
        }
        if (size == 1) {
            return this.decodeSingleValue(type);
        }
        ArrayList<Object> ints = new ArrayList<Object>(size);
        for (int i = 0; i < size; ++i) {
            Object val = this.decodeSingleValue(type);
            if (val == null) continue;
            ints.add(val);
        }
        return ints.isEmpty() ? null : ints;
    }

    public final Object decodeSingleValue(BCF2Type type) throws IOException {
        int value = this.decodeInt(type);
        if (value == type.getMissingBytes()) {
            return null;
        }
        switch (type) {
            case INT8: 
            case INT16: 
            case INT32: {
                return value;
            }
            case FLOAT: {
                return this.rawFloatToFloat(value);
            }
            case CHAR: {
                return value & 0xFF;
            }
        }
        throw new ReviewedStingException("BCF2 codec doesn't know how to decode type " + (Object)((Object)type));
    }

    private final Object decodeLiteralString(int size) {
        assert (size > 0);
        byte[] bytes = new byte[size];
        try {
            int goodLength;
            this.recordStream.read(bytes);
            for (goodLength = 0; goodLength < bytes.length && bytes[goodLength] != 0; ++goodLength) {
            }
            if (goodLength == 0) {
                return null;
            }
            String s = new String(bytes, 0, goodLength);
            return BCF2Utils.isCollapsedString(s) ? BCF2Utils.explodeStringList(s) : s;
        }
        catch (IOException e) {
            throw new ReviewedStingException("readByte failure", e);
        }
    }

    @Ensures(value={"result >= 0"})
    public final int decodeNumberOfElements(byte typeDescriptor) throws IOException {
        if (BCF2Utils.sizeIsOverflow(typeDescriptor)) {
            return this.decodeInt(this.readTypeDescriptor(), -1);
        }
        return BCF2Utils.decodeSize(typeDescriptor);
    }

    @Requires(value={"BCF2Utils.decodeSize(typeDescriptor) == 1"})
    public final int decodeInt(byte typeDescriptor, int missingValue) throws IOException {
        BCF2Type type = BCF2Utils.decodeType(typeDescriptor);
        int i = this.decodeInt(type);
        return i == type.getMissingBytes() ? missingValue : i;
    }

    @Requires(value={"type != null"})
    public final int decodeInt(BCF2Type type) throws IOException {
        return type.read(this.recordStream);
    }

    @Requires(value={"type != null", "type.isIntegerType()", "size >= 0"})
    public final int[] decodeIntArray(int size, BCF2Type type, int[] maybeDest) throws IOException {
        int val1;
        if (size == 0) {
            return null;
        }
        if (maybeDest != null && maybeDest.length < size) {
            maybeDest = null;
        }
        if ((val1 = this.decodeInt(type)) == type.getMissingBytes()) {
            for (int i = 1; i < size; ++i) {
                this.decodeInt(type);
            }
            return null;
        }
        int[] ints = maybeDest == null ? new int[size] : maybeDest;
        ints[0] = val1;
        for (int i = 1; i < size; ++i) {
            ints[i] = this.decodeInt(type);
            if (ints[i] != type.getMissingBytes()) continue;
            for (int j = i + 1; j < size; ++j) {
                this.decodeInt(type);
            }
            return Arrays.copyOf(ints, i);
        }
        return ints;
    }

    public final int[] decodeIntArray(byte typeDescriptor, int size) throws IOException {
        BCF2Type type = BCF2Utils.decodeType(typeDescriptor);
        return this.decodeIntArray(size, type, null);
    }

    private double rawFloatToFloat(int rawFloat) {
        return Float.intBitsToFloat(rawFloat);
    }

    public final int readBlockSize(InputStream inputStream) throws IOException {
        return BCF2Type.INT32.read(inputStream);
    }

    @Requires(value={"blockSizeInBytes >= 0", "inputStream != null"})
    @Ensures(value={"result != null"})
    private static byte[] readRecordBytes(int blockSizeInBytes, InputStream inputStream) {
        assert (blockSizeInBytes >= 0);
        byte[] record = new byte[blockSizeInBytes];
        try {
            int bytesRead = 0;
            int nReadAttempts = 0;
            while (bytesRead < blockSizeInBytes) {
                int read1 = inputStream.read(record, bytesRead, blockSizeInBytes - bytesRead);
                if (read1 == -1) {
                    BCF2Decoder.validateReadBytes(bytesRead, nReadAttempts, blockSizeInBytes);
                    continue;
                }
                bytesRead += read1;
            }
            if (nReadAttempts > 1) {
                logger.warn("Required multiple read attempts to actually get the entire BCF2 block, unexpected behavior");
            }
            BCF2Decoder.validateReadBytes(bytesRead, nReadAttempts, blockSizeInBytes);
        }
        catch (IOException e) {
            throw new UserException.CouldNotReadInputFile("I/O error while reading BCF2 file", (Exception)e);
        }
        return record;
    }

    private static void validateReadBytes(int actuallyRead, int nReadAttempts, int expected) {
        assert (expected >= 0);
        if (actuallyRead < expected) {
            throw new UserException.MalformedBCF2(String.format("Failed to read next complete record: expected %d bytes but read only %d after %d iterations", expected, actuallyRead, nReadAttempts));
        }
    }

    public final byte readTypeDescriptor() throws IOException {
        return BCF2Utils.readByte(this.recordStream);
    }
}

