/*
 * Decompiled with CFR 0.152.
 */
package picard.analysis;

import htsjdk.samtools.SamReader;
import htsjdk.samtools.SamReaderFactory;
import htsjdk.samtools.filter.SamRecordFilter;
import htsjdk.samtools.filter.SecondaryAlignmentFilter;
import htsjdk.samtools.metrics.MetricBase;
import htsjdk.samtools.metrics.MetricsFile;
import htsjdk.samtools.reference.ReferenceSequence;
import htsjdk.samtools.reference.ReferenceSequenceFileWalker;
import htsjdk.samtools.util.Histogram;
import htsjdk.samtools.util.IOUtil;
import htsjdk.samtools.util.Log;
import htsjdk.samtools.util.ProgressLogger;
import htsjdk.samtools.util.SamLocusIterator;
import java.io.File;
import java.util.ArrayList;
import java.util.HashSet;
import picard.analysis.CountingDuplicateFilter;
import picard.analysis.CountingFilter;
import picard.analysis.CountingMapQFilter;
import picard.analysis.CountingPairedFilter;
import picard.cmdline.CommandLineProgram;
import picard.cmdline.CommandLineProgramProperties;
import picard.cmdline.Option;
import picard.cmdline.programgroups.Metrics;
import picard.util.MathUtil;

@CommandLineProgramProperties(usage="Computes a number of metrics that are useful for evaluating coverage and performance of whole genome sequencing experiments.", usageShort="Writes whole genome sequencing-related metrics for a SAM or BAM file", programGroup=Metrics.class)
public class CollectWgsMetrics
extends CommandLineProgram {
    @Option(shortName="I", doc="Input SAM or BAM file.")
    public File INPUT;
    @Option(shortName="O", doc="Output metrics file.")
    public File OUTPUT;
    @Option(shortName="R", doc="The reference sequence fasta aligned to.")
    public File REFERENCE_SEQUENCE;
    @Option(shortName="MQ", doc="Minimum mapping quality for a read to contribute coverage.", overridable=true)
    public int MINIMUM_MAPPING_QUALITY = 20;
    @Option(shortName="Q", doc="Minimum base quality for a base to contribute coverage.", overridable=true)
    public int MINIMUM_BASE_QUALITY = 20;
    @Option(shortName="CAP", doc="Treat bases with coverage exceeding this value as if they had coverage at this value.", overridable=true)
    public int COVERAGE_CAP = 250;
    @Option(doc="For debugging purposes, stop after processing this many genomic bases.")
    public long STOP_AFTER = -1L;
    @Option(doc="Determines whether to include the base quality histogram in the metrics file.")
    public boolean INCLUDE_BQ_HISTOGRAM = false;
    @Option(doc="If true, count unpaired reads, and paired reads with one end unmapped")
    public boolean COUNT_UNPAIRED = false;
    private final Log log = Log.getInstance(CollectWgsMetrics.class);

    public static void main(String[] args) {
        new CollectWgsMetrics().instanceMainWithExit(args);
    }

    @Override
    protected int doWork() {
        IOUtil.assertFileIsReadable(this.INPUT);
        IOUtil.assertFileIsWritable(this.OUTPUT);
        IOUtil.assertFileIsReadable(this.REFERENCE_SEQUENCE);
        ProgressLogger progress = new ProgressLogger(this.log, 10000000, "Processed", "loci");
        ReferenceSequenceFileWalker refWalker = new ReferenceSequenceFileWalker(this.REFERENCE_SEQUENCE);
        SamReader in = SamReaderFactory.makeDefault().referenceSequence(this.REFERENCE_SEQUENCE).open(this.INPUT);
        SamLocusIterator iterator = this.getLocusIterator(in);
        ArrayList<SamRecordFilter> filters = new ArrayList<SamRecordFilter>();
        CountingDuplicateFilter dupeFilter = new CountingDuplicateFilter();
        CountingMapQFilter mapqFilter = new CountingMapQFilter(this.MINIMUM_MAPPING_QUALITY);
        CountingPairedFilter pairFilter = new CountingPairedFilter();
        filters.add(mapqFilter);
        filters.add(dupeFilter);
        if (!this.COUNT_UNPAIRED) {
            filters.add(pairFilter);
        }
        filters.add(new SecondaryAlignmentFilter());
        iterator.setSamFilters(filters);
        iterator.setEmitUncoveredLoci(true);
        iterator.setMappingQualityScoreCutoff(0);
        iterator.setQualityScoreCutoff(0);
        iterator.setIncludeNonPfReads(false);
        int max = this.COVERAGE_CAP;
        long[] HistogramArray = new long[max + 1];
        long[] baseQHistogramArray = new long[127];
        boolean usingStopAfter = this.STOP_AFTER > 0L;
        long stopAfter = this.STOP_AFTER - 1L;
        long counter = 0L;
        long basesExcludedByBaseq = 0L;
        long basesExcludedByOverlap = 0L;
        long basesExcludedByCapping = 0L;
        while (iterator.hasNext()) {
            SamLocusIterator.LocusInfo info = iterator.next();
            ReferenceSequence ref = refWalker.get(info.getSequenceIndex());
            byte base = ref.getBases()[info.getPosition() - 1];
            if (base == 78) continue;
            HashSet<String> readNames = new HashSet<String>(info.getRecordAndPositions().size());
            int pileupSize = 0;
            for (SamLocusIterator.RecordAndOffset recs : info.getRecordAndPositions()) {
                if (recs.getBaseQuality() < this.MINIMUM_BASE_QUALITY) {
                    ++basesExcludedByBaseq;
                    continue;
                }
                if (!readNames.add(recs.getRecord().getReadName())) {
                    ++basesExcludedByOverlap;
                    continue;
                }
                if (++pileupSize > max) continue;
                byte by = recs.getRecord().getBaseQualities()[recs.getOffset()];
                baseQHistogramArray[by] = baseQHistogramArray[by] + 1L;
            }
            int depth = Math.min(readNames.size(), max);
            if (depth < readNames.size()) {
                basesExcludedByCapping += (long)(readNames.size() - max);
            }
            int n = depth;
            HistogramArray[n] = HistogramArray[n] + 1L;
            progress.record(info.getSequenceName(), info.getPosition());
            if (!usingStopAfter || ++counter <= stopAfter) continue;
            break;
        }
        Histogram<Integer> histo = new Histogram<Integer>("coverage", "count");
        for (int i = 0; i < HistogramArray.length; ++i) {
            histo.increment(i, HistogramArray[i]);
        }
        Histogram<Integer> baseQHisto = new Histogram<Integer>("value", "baseq_count");
        for (int i = 0; i < baseQHistogramArray.length; ++i) {
            baseQHisto.increment(i, baseQHistogramArray[i]);
        }
        WgsMetrics metrics = this.generateWgsMetrics();
        metrics.GENOME_TERRITORY = (long)histo.getSumOfValues();
        metrics.MEAN_COVERAGE = histo.getMean();
        metrics.SD_COVERAGE = histo.getStandardDeviation();
        metrics.MEDIAN_COVERAGE = histo.getMedian();
        metrics.MAD_COVERAGE = histo.getMedianAbsoluteDeviation();
        long basesExcludedByDupes = this.getBasesExcludedBy(dupeFilter);
        long basesExcludedByMapq = this.getBasesExcludedBy(mapqFilter);
        long basesExcludedByPairing = this.getBasesExcludedBy(pairFilter);
        double total = histo.getSum();
        double totalWithExcludes = total + (double)basesExcludedByDupes + (double)basesExcludedByMapq + (double)basesExcludedByPairing + (double)basesExcludedByBaseq + (double)basesExcludedByOverlap + (double)basesExcludedByCapping;
        metrics.PCT_EXC_DUPE = (double)basesExcludedByDupes / totalWithExcludes;
        metrics.PCT_EXC_MAPQ = (double)basesExcludedByMapq / totalWithExcludes;
        metrics.PCT_EXC_UNPAIRED = (double)basesExcludedByPairing / totalWithExcludes;
        metrics.PCT_EXC_BASEQ = (double)basesExcludedByBaseq / totalWithExcludes;
        metrics.PCT_EXC_OVERLAP = (double)basesExcludedByOverlap / totalWithExcludes;
        metrics.PCT_EXC_CAPPED = (double)basesExcludedByCapping / totalWithExcludes;
        metrics.PCT_EXC_TOTAL = (totalWithExcludes - total) / totalWithExcludes;
        metrics.PCT_5X = (double)MathUtil.sum(HistogramArray, 5, HistogramArray.length) / (double)metrics.GENOME_TERRITORY;
        metrics.PCT_10X = (double)MathUtil.sum(HistogramArray, 10, HistogramArray.length) / (double)metrics.GENOME_TERRITORY;
        metrics.PCT_15X = (double)MathUtil.sum(HistogramArray, 15, HistogramArray.length) / (double)metrics.GENOME_TERRITORY;
        metrics.PCT_20X = (double)MathUtil.sum(HistogramArray, 20, HistogramArray.length) / (double)metrics.GENOME_TERRITORY;
        metrics.PCT_25X = (double)MathUtil.sum(HistogramArray, 25, HistogramArray.length) / (double)metrics.GENOME_TERRITORY;
        metrics.PCT_30X = (double)MathUtil.sum(HistogramArray, 30, HistogramArray.length) / (double)metrics.GENOME_TERRITORY;
        metrics.PCT_40X = (double)MathUtil.sum(HistogramArray, 40, HistogramArray.length) / (double)metrics.GENOME_TERRITORY;
        metrics.PCT_50X = (double)MathUtil.sum(HistogramArray, 50, HistogramArray.length) / (double)metrics.GENOME_TERRITORY;
        metrics.PCT_60X = (double)MathUtil.sum(HistogramArray, 60, HistogramArray.length) / (double)metrics.GENOME_TERRITORY;
        metrics.PCT_70X = (double)MathUtil.sum(HistogramArray, 70, HistogramArray.length) / (double)metrics.GENOME_TERRITORY;
        metrics.PCT_80X = (double)MathUtil.sum(HistogramArray, 80, HistogramArray.length) / (double)metrics.GENOME_TERRITORY;
        metrics.PCT_90X = (double)MathUtil.sum(HistogramArray, 90, HistogramArray.length) / (double)metrics.GENOME_TERRITORY;
        metrics.PCT_100X = (double)MathUtil.sum(HistogramArray, 100, HistogramArray.length) / (double)metrics.GENOME_TERRITORY;
        MetricsFile out = this.getMetricsFile();
        out.addMetric(metrics);
        out.addHistogram(histo);
        if (this.INCLUDE_BQ_HISTOGRAM) {
            out.addHistogram(baseQHisto);
        }
        out.write(this.OUTPUT);
        return 0;
    }

    protected WgsMetrics generateWgsMetrics() {
        return new WgsMetrics();
    }

    protected long getBasesExcludedBy(CountingFilter filter) {
        return filter.getFilteredBases();
    }

    protected SamLocusIterator getLocusIterator(SamReader in) {
        return new SamLocusIterator(in);
    }

    public static class WgsMetrics
    extends MetricBase {
        public long GENOME_TERRITORY;
        public double MEAN_COVERAGE;
        public double SD_COVERAGE;
        public double MEDIAN_COVERAGE;
        public double MAD_COVERAGE;
        public double PCT_EXC_MAPQ;
        public double PCT_EXC_DUPE;
        public double PCT_EXC_UNPAIRED;
        public double PCT_EXC_BASEQ;
        public double PCT_EXC_OVERLAP;
        public double PCT_EXC_CAPPED;
        public double PCT_EXC_TOTAL;
        public double PCT_5X;
        public double PCT_10X;
        public double PCT_15X;
        public double PCT_20X;
        public double PCT_25X;
        public double PCT_30X;
        public double PCT_40X;
        public double PCT_50X;
        public double PCT_60X;
        public double PCT_70X;
        public double PCT_80X;
        public double PCT_90X;
        public double PCT_100X;
    }
}

