#!/usr/bin/env python
import argparse
import os
import shutil
import subprocess
import sys
import tempfile

BUFF_SIZE = 1048576
OUTPUT_DIR = 'phylogenomicsAnalysis_dir'

parser = argparse.ArgumentParser()

parser.add_argument('--orthogroup_faa', dest='orthogroup_faa', help="Input dataset files_path")
parser.add_argument('--scaffold', dest='scaffold', default='mode', help='Orthogroups or gene families proteins scaffold')
parser.add_argument('--method', dest='method', help='Protein clustering method')
parser.add_argument('--config_dir', dest='config_dir', help='Directory containing default configuration files')
parser.add_argument('--num_threads', dest='num_threads', type=int, help='Number of threads to use for execution')
parser.add_argument('--orthogroup_fna', dest='orthogroup_fna', default=None, help="Flag for coding sequences associated with orthogroups")
parser.add_argument('--sequence_type', dest='sequence_type', default=None, help="Sequence type used in the phylogenetic inference")
parser.add_argument('--alignments_method', dest='alignments_method', default=None, help='Multiple sequence alignments method')
parser.add_argument('--pasta_script_path', dest='pasta_script_path', default=None, help='Path to script for executing pasta')
parser.add_argument('--pasta_iter_limit', dest='pasta_iter_limit', type=int, default=None, help='"Maximum number of iteration that the PASTA algorithm will execute')
parser.add_argument('--tree_inference', dest='tree_inference', default=None, help='Phylogenetic trees inference method')
parser.add_argument('--rooting_order', dest='rooting_order', default=None, help='Rooting order configuration for rooting trees')
parser.add_argument('--bootstrap_replicates', dest='bootstrap_replicates', type=int, default=None, help='Number of replicates for rapid bootstrap analysis')
parser.add_argument('--max_orthogroup_size', dest='max_orthogroup_size', type=int, default=None, help='Maximum number of sequences in orthogroup alignments')
parser.add_argument('--min_orthogroup_size', dest='min_orthogroup_size', type=int, default=None, help='Minimum number of sequences in orthogroup alignments')
parser.add_argument('--remove_sequences', dest='remove_sequences', default=None, type=float, help='Remove sequences with gaps of')
parser.add_argument('--trim_type', dest='trim_type', default=None, help='Process used for gap trimming')
parser.add_argument('--gap_trimming', dest='gap_trimming', default=None, type=float, help='Remove sites in alignments with gaps of')
parser.add_argument('--output_ptortho', dest='output_ptortho', default=None, help='Output for orthogroups')
parser.add_argument('--output_ptortho_dir', dest='output_ptortho_dir', default=None, help='output_ptortho.files_path')
parser.add_argument('--output_ptorthocs', dest='output_ptorthocs', default=None, help='Output for orthogroups with corresponding coding sequences')
parser.add_argument('--output_ptorthocs_dir', dest='output_ptorthocs_dir', default=None, help='output_ptorthocs.files_path')
parser.add_argument('--output_aln', dest='output_aln', default=None, help='Output for orthogroups alignments')
parser.add_argument('--output_aln_dir', dest='output_aln_dir', default=None, help='output_aln.files_path')
parser.add_argument('--output_tree', dest='output_tree', default=None, help='Output for phylogenetic trees')
parser.add_argument('--output_tree_dir', dest='output_tree_dir', default=None, help='output_tree.files_path')

args = parser.parse_args()


def get_stderr_exception(tmp_err, tmp_stderr, tmp_out, tmp_stdout, include_stdout=False):
    tmp_stderr.close()
    # Get stderr, allowing for case where it's very large.
    tmp_stderr = open(tmp_err, 'rb')
    stderr_str = ''
    buffsize = BUFF_SIZE
    try:
        while True:
            stderr_str += tmp_stderr.read(buffsize)
            if not stderr_str or len(stderr_str) % buffsize != 0:
                break
    except OverflowError:
        pass
    tmp_stderr.close()
    if include_stdout:
        tmp_stdout = open(tmp_out, 'rb')
        stdout_str = ''
        buffsize = BUFF_SIZE
        try:
            while True:
                stdout_str += tmp_stdout.read(buffsize)
                if not stdout_str or len(stdout_str) % buffsize != 0:
                    break
        except OverflowError:
            pass
    tmp_stdout.close()
    if include_stdout:
        return 'STDOUT\n%s\n\nSTDERR\n%s\n' % (stdout_str, stderr_str)
    return stderr_str


def move_directory_files(source_dir, destination_dir):
    source_directory = os.path.abspath(source_dir)
    destination_directory = os.path.abspath(destination_dir)
    if not os.path.isdir(destination_directory):
        os.makedirs(destination_directory)
    for dir_entry in os.listdir(source_directory):
        source_entry = os.path.join(source_directory, dir_entry)
        shutil.move(source_entry, destination_directory)


def stop_err(msg):
    sys.stderr.write(msg)
    sys.exit(1)


def write_html_output(output, title, dir):
    with open(output, 'w') as fh:
        fh.write('<html><head><h3>%s</h3></head>\n' % title)
        fh.write('<body><p/><table cellpadding="2">\n')
        fh.write('<tr><th>Size</th><th>Name</th></tr>\n')
        for index, fname in enumerate(sorted(os.listdir(dir))):
            if index % 2 == 0:
                bgcolor = '#D8D8D8'
            else:
                bgcolor = '#FFFFFF'
            try:
                size = str(os.path.getsize(os.path.join(dir, fname)))
            except:
                size = 'unknown'
            link = '<a href="%s" type="text/plain">%s</a>\n' % (fname, fname)
            fh.write('<tr bgcolor="%s"><td>%s</td><td>%s</td></tr>\n' % (bgcolor, size, link))
        fh.write('</table></body></html>\n')


# Define command response buffers.
tmp_out = tempfile.NamedTemporaryFile().name
tmp_stdout = open(tmp_out, 'wb')
tmp_err = tempfile.NamedTemporaryFile().name
tmp_stderr = open(tmp_err, 'wb')
# Build the command line.
cmd = 'PhylogenomicsAnalysis'
cmd += ' --orthogroup_faa %s' % args.orthogroup_faa
cmd += ' --scaffold %s' % args.scaffold
cmd += ' --method %s' % args.method
cmd += ' --config_dir %s' % args.config_dir
cmd += ' --num_threads %d' % args.num_threads

if args.orthogroup_fna is not None:
    cmd += ' --orthogroup_fna'
if args.sequence_type is not None:
    cmd += ' --sequence_type %s' % args.sequence_type
if args.alignments_method is not None:
    if args.alignments_method == 'create_alignments':
        cmd += ' --create_alignments'
    elif args.alignments_method == 'add_alignments':
        cmd += ' --add_alignments'
    elif args.alignments_method == 'pasta_alignments':
        cmd += ' --pasta_alignments'
        cmd += ' --pasta_script_path %s' % args.pasta_script_path
        cmd += ' --pasta_iter_limit %d' % args.pasta_iter_limit
if args.tree_inference is not None:
    cmd += ' --tree_inference %s' % args.tree_inference
    if args.tree_inference == 'raxml':
        if args.rooting_order is not None:
            cmd += ' --rooting_order %s' % args.rooting_order
        cmd += ' --bootstrap_replicates %d' % args.bootstrap_replicates
if args.max_orthogroup_size is not None:
    cmd += ' --max_orthogroup_size %d' % args.max_orthogroup_size
if args.min_orthogroup_size is not None:
    cmd += ' --min_orthogroup_size %d' % args.min_orthogroup_size
if args.remove_sequences is not None:
    cmd += ' --remove_sequences %4f' % args.remove_sequences
if args.trim_type is not None:
    if args.trim_type == 'automated_trimming':
        cmd += ' --automated_trimming'
    else:
        cmd += ' --gap_trimming %4f' % args.gap_trimming
# Run the command.
proc = subprocess.Popen(args=cmd, stderr=tmp_stderr, stdout=tmp_stdout, shell=True)
rc = proc.wait()
# Handle execution errors.
if rc != 0:
    error_message = get_stderr_exception(tmp_err, tmp_stderr, tmp_out, tmp_stdout)
    stop_err( error_message )
# Handle phylogenies for orthogroups outputs.
if args.orthogroup_fna is not None:
    out_file = args.output_ptorthocs
    orthogroups_dest_dir = args.output_ptorthocs_dir
    title = 'Phylogenies files for orthogroups and corresponding coding sequences'
else:
    out_file = args.output_ptortho
    orthogroups_dest_dir = args.output_ptortho_dir
    title = 'Phylogenies files for orthogroups'
orthogroups_src_dir = os.path.join(OUTPUT_DIR, 'orthogroups_fasta')
move_directory_files(orthogroups_src_dir, orthogroups_dest_dir)
write_html_output(out_file, title, orthogroups_dest_dir)
# Handle multiple sequences alignments for orthogroups outputs.
if args.output_aln is not None:
    alignments_src_dir = os.path.join(OUTPUT_DIR, 'orthogroups_aln')
    alignments_dest_dir = args.output_aln_dir
    title = 'Multiple sequence alignments files for orthogroups'
    move_directory_files(alignments_src_dir, alignments_dest_dir)
    write_html_output(args.output_aln, title, alignments_dest_dir)
# Handle phylogenies for orthogroups outputs.
if args.output_tree is not None:
    trees_src_dir = os.path.join(OUTPUT_DIR, 'orthogroups_tree')
    trees_dest_dir = args.output_tree_dir
    title = 'Phylogenetic tree files for orthogroups'
    move_directory_files(trees_src_dir, trees_dest_dir)
    write_html_output(args.output_tree, title, trees_dest_dir)
