#!/usr/bin/env Rscript

suppressPackageStartupMessages(library("data.table"))
suppressPackageStartupMessages(library("optparse"))

option_list <- list(
            make_option(c("--input_dir"), action="store", dest="input_dir", help="Directory containing .csv outputs from insect_phenology_model"),
            make_option(c("--end_date"), action="store", dest="end_date", help="End date for date interval"),
            make_option(c("--start_date"), action="store", dest="start_date", help="Start date for date interval"),
            make_option(c("--script_dir"), action="store", dest="script_dir", help="R script source directory")
)

parser <- OptionParser(usage="%prog [options] file", option_list=option_list);
args <- parse_args(parser, positional_arguments=TRUE);
opt <- args$options;

get_new_temperature_data_frame = function(input_file) {
    temperature_data_frame = read.csv(file=input_file, header=T, strip.white=TRUE, stringsAsFactors=FALSE, sep=",");
    return(temperature_data_frame);
}

# Import the shared utility functions.
utils_path <- paste(opt$script_dir, "utils.R", sep="/");
source(utils_path);

# FIXME: currently custom date fields are free text, but
# Galaxy should soon include support for a date selector
# at which point this tool should be enhanced to use it.
# Validate start_date.
start_date = format(opt$start_date);
end_date = format(opt$end_date);

# Calaculate the number of days in the date interval.
start_date = validate_date(start_date);
# Validate end_date.
end_date = validate_date(end_date);
if (start_date >= end_date) {
    stop_err("The start date must be between 1 and 50 days before the end date when setting date intervals for plots.");
}
# Calculate the number of days in the date interval.
num_days = difftime(end_date, start_date, units=c("days"));
# Add 1 to the number of days to make the dates inclusive.  For
# example, if the user enters a date range of 2018-01-01 to
# 2018-01-31, they likely expect the end date to be included.
num_days = num_days + 1;
if (num_days > 50) {
    # We need to restrict date intervals since
    # plots render tick marks for each day.
    stop_err("Date intervals for plotting cannot exceed 50 days.");
}
# Display the total number of days in the Galaxy history item blurb.
cat("Number of days in date interval: ", num_days, "\n");

input_files = list.files(path=opt$input_dir, full.names=TRUE);
for(input_file in input_files) {
    temperature_data_frame = get_new_temperature_data_frame(input_file);
    start_date_row = which(temperature_data_frame$DATE==start_date);
    end_date_row = which(temperature_data_frame$DATE==end_date);
    # Extract the date interval.
    temperature_data_frame = temperature_data_frame[start_date_row:end_date_row,];
    # Get the ticks date labels for plots.
        ticks_and_labels = get_x_axis_ticks_and_labels(temperature_data_frame, date_interval=TRUE);
        ticks = c(unlist(ticks_and_labels[1]));
        date_labels = c(unlist(ticks_and_labels[2]));
    # All latitude values are the same, so get the value
    # for plots from the first row.
    latitude = temperature_data_frame$LATITUDE[1];
    # Save the date interval data into an output file
    # named the same as the input.
    file_path = paste("output_data_dir", basename(input_file), sep="/");
    write.csv(temperature_data_frame, file=file_path, row.names=F);
    # TODO: Save the date interval plots...
}
