/*
 * Decompiled with CFR 0.152.
 */
package picard.vcf;

import htsjdk.samtools.Defaults;
import htsjdk.samtools.SAMSequenceRecord;
import htsjdk.samtools.liftover.LiftOver;
import htsjdk.samtools.reference.ReferenceSequenceFileWalker;
import htsjdk.samtools.util.CloserUtil;
import htsjdk.samtools.util.CollectionUtil;
import htsjdk.samtools.util.IOUtil;
import htsjdk.samtools.util.Interval;
import htsjdk.samtools.util.Log;
import htsjdk.samtools.util.ProgressLogger;
import htsjdk.samtools.util.SequenceUtil;
import htsjdk.samtools.util.SortingCollection;
import htsjdk.samtools.util.StringUtil;
import htsjdk.variant.variantcontext.Allele;
import htsjdk.variant.variantcontext.Genotype;
import htsjdk.variant.variantcontext.GenotypeBuilder;
import htsjdk.variant.variantcontext.GenotypesContext;
import htsjdk.variant.variantcontext.VariantContext;
import htsjdk.variant.variantcontext.VariantContextBuilder;
import htsjdk.variant.variantcontext.writer.Options;
import htsjdk.variant.variantcontext.writer.VariantContextWriter;
import htsjdk.variant.variantcontext.writer.VariantContextWriterBuilder;
import htsjdk.variant.vcf.VCFFileReader;
import htsjdk.variant.vcf.VCFFilterHeaderLine;
import htsjdk.variant.vcf.VCFHeader;
import htsjdk.variant.vcf.VCFRecordCodec;
import java.io.File;
import java.text.DecimalFormat;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import picard.cmdline.CommandLineProgram;
import picard.cmdline.CommandLineProgramProperties;
import picard.cmdline.Option;
import picard.cmdline.programgroups.VcfOrBcf;

@CommandLineProgramProperties(usage="Lifts a VCF over from one genome build to another using UCSC liftover. The output file will be sorted and indexed. Records may be rejected because they cannot be lifted over or because post-liftover the reference allele mismatches the target genome build.  Rejected records will be emitted with filters to the REJECT file, on the source genome.", usageShort="Lifts a VCF between genome builds.", programGroup=VcfOrBcf.class)
public class LiftoverVcf
extends CommandLineProgram {
    @Option(shortName="I", doc="The input VCF/BCF file to be lifted over.")
    public File INPUT;
    @Option(shortName="O", doc="The output location to write the lifted over VCF/BCF to.")
    public File OUTPUT;
    @Option(shortName="C", doc="The liftover chain file. See https://genome.ucsc.edu/goldenPath/help/chain.html for a description of chain files.  See http://hgdownload.soe.ucsc.edu/downloads.html#terms for where to download chain files.")
    public File CHAIN;
    @Option(doc="File to which to write rejected records.")
    public File REJECT;
    @Option(shortName="R", common=false, doc="The reference sequence (fasta) for the TARGET genome build.  The fasta file must have an accompanying sqeuence dictionary (.dict file).")
    public File REFERENCE_SEQUENCE = Defaults.REFERENCE_FASTA;
    public static final String FILTER_CANNOT_LIFTOVER_INDEL = "ReverseComplementedIndel";
    public static final String FILTER_NO_TARGET = "NoTarget";
    public static final String FILTER_MISMATCHING_REF_ALLELE = "MismatchedRefAllele";
    private static final List<VCFFilterHeaderLine> FILTERS = CollectionUtil.makeList(new VCFFilterHeaderLine("ReverseComplementedIndel", "Indel falls into a reverse complemented region in the target genome."), new VCFFilterHeaderLine("NoTarget", "Variant could not be lifted between genome builds."), new VCFFilterHeaderLine("MismatchedRefAllele", "Reference allele does not match reference genome sequence after liftover."));
    private final Log log = Log.getInstance(LiftoverVcf.class);

    public static void main(String[] args) {
        new LiftoverVcf().instanceMainWithExit(args);
    }

    @Override
    protected int doWork() {
        IOUtil.assertFileIsReadable(this.INPUT);
        IOUtil.assertFileIsReadable(this.REFERENCE_SEQUENCE);
        IOUtil.assertFileIsReadable(this.CHAIN);
        IOUtil.assertFileIsWritable(this.OUTPUT);
        IOUtil.assertFileIsWritable(this.REJECT);
        LiftOver liftOver = new LiftOver(this.CHAIN);
        VCFFileReader in = new VCFFileReader(this.INPUT, false);
        this.log.info("Loading up the target reference genome.");
        ReferenceSequenceFileWalker walker = new ReferenceSequenceFileWalker(this.REFERENCE_SEQUENCE);
        HashMap<String, byte[]> refSeqs = new HashMap<String, byte[]>();
        for (SAMSequenceRecord rec : walker.getSequenceDictionary().getSequences()) {
            refSeqs.put(rec.getSequenceName(), walker.get(rec.getSequenceIndex()).getBases());
        }
        CloserUtil.close(walker);
        VCFHeader inHeader = in.getFileHeader();
        VCFHeader outHeader = new VCFHeader(inHeader);
        outHeader.setSequenceDictionary(walker.getSequenceDictionary());
        VariantContextWriter out = new VariantContextWriterBuilder().setOption(Options.INDEX_ON_THE_FLY).setOutputFile(this.OUTPUT).setReferenceDictionary(walker.getSequenceDictionary()).build();
        out.writeHeader(outHeader);
        VariantContextWriter rejects = new VariantContextWriterBuilder().setOutputFile(this.REJECT).unsetOption(Options.INDEX_ON_THE_FLY).build();
        VCFHeader rejectHeader = new VCFHeader(in.getFileHeader());
        for (VCFFilterHeaderLine line : FILTERS) {
            rejectHeader.addMetaDataLine(line);
        }
        rejects.writeHeader(rejectHeader);
        long failedLiftover = 0L;
        long failedAlleleCheck = 0L;
        long total = 0L;
        this.log.info("Lifting variants over and sorting.");
        SortingCollection<VariantContext> sorter = SortingCollection.newInstance(VariantContext.class, new VCFRecordCodec(outHeader), outHeader.getVCFRecordComparator(), (int)this.MAX_RECORDS_IN_RAM, this.TMP_DIR);
        ProgressLogger progress = new ProgressLogger(this.log, 1000000, "read");
        HashMap<Allele, Allele> reverseComplementAlleleMap = new HashMap<Allele, Allele>(10);
        for (VariantContext ctx : in) {
            ++total;
            Interval source = new Interval(ctx.getContig(), ctx.getStart(), ctx.getEnd(), false, ctx.getContig() + ":" + ctx.getStart() + "-" + ctx.getEnd());
            Interval target = liftOver.liftOver(source, 1.0);
            if (target == null || target.isNegativeStrand() && (ctx.isMixed() || ctx.isIndel())) {
                String reason = target == null ? FILTER_NO_TARGET : FILTER_CANNOT_LIFTOVER_INDEL;
                rejects.add(new VariantContextBuilder(ctx).filter(reason).make());
                ++failedLiftover;
            } else {
                reverseComplementAlleleMap.clear();
                ArrayList<Allele> alleles = new ArrayList<Allele>();
                for (Allele oldAllele : ctx.getAlleles()) {
                    if (target.isPositiveStrand() || oldAllele.isSymbolic()) {
                        alleles.add(oldAllele);
                        continue;
                    }
                    Allele fixedAllele = Allele.create(SequenceUtil.reverseComplement(oldAllele.getBaseString()), oldAllele.isReference());
                    alleles.add(fixedAllele);
                    reverseComplementAlleleMap.put(oldAllele, fixedAllele);
                }
                VariantContextBuilder builder = new VariantContextBuilder(ctx.getSource(), target.getContig(), target.getStart(), target.getEnd(), alleles);
                builder.id(ctx.getID());
                builder.attributes(ctx.getAttributes());
                builder.genotypes(LiftoverVcf.fixGenotypes(ctx.getGenotypes(), reverseComplementAlleleMap));
                builder.filters(ctx.getFilters());
                builder.log10PError(ctx.getLog10PError());
                boolean mismatchesReference = false;
                for (Allele allele : builder.getAlleles()) {
                    if (!allele.isReference()) continue;
                    byte[] ref = (byte[])refSeqs.get(target.getContig());
                    String refString = StringUtil.bytesToString(ref, target.getStart() - 1, target.length());
                    if (refString.equalsIgnoreCase(allele.getBaseString())) break;
                    mismatchesReference = true;
                    break;
                }
                if (mismatchesReference) {
                    rejects.add(new VariantContextBuilder(ctx).filter(FILTER_MISMATCHING_REF_ALLELE).make());
                    ++failedAlleleCheck;
                } else {
                    sorter.add(builder.make());
                }
            }
            progress.record(ctx.getContig(), ctx.getStart());
        }
        DecimalFormat pfmt = new DecimalFormat("0.0000%");
        String pct = pfmt.format((double)(failedLiftover + failedAlleleCheck) / (double)total);
        this.log.info("Processed ", total, " variants.");
        this.log.info(failedLiftover, " variants failed to liftover.");
        this.log.info(failedAlleleCheck, " variants lifted over but had mismatching reference alleles after lift over.");
        this.log.info(pct, " of variants were not successfully lifted over and written to the output.");
        rejects.close();
        in.close();
        sorter.doneAdding();
        progress = new ProgressLogger(this.log, 1000000, "written");
        this.log.info("Writing out sorted records to final VCF.");
        for (VariantContext ctx : sorter) {
            out.add(ctx);
            progress.record(ctx.getContig(), ctx.getStart());
        }
        out.close();
        sorter.cleanup();
        return 0;
    }

    protected static GenotypesContext fixGenotypes(GenotypesContext originals, Map<Allele, Allele> reverseComplementAlleleMap) {
        if (reverseComplementAlleleMap.isEmpty()) {
            return originals;
        }
        GenotypesContext fixedGenotypes = GenotypesContext.create(originals.size());
        for (Genotype genotype : originals) {
            ArrayList<Allele> fixedAlleles = new ArrayList<Allele>();
            for (Allele allele : genotype.getAlleles()) {
                Allele fixedAllele = reverseComplementAlleleMap.containsKey(allele) ? reverseComplementAlleleMap.get(allele) : allele;
                fixedAlleles.add(fixedAllele);
            }
            fixedGenotypes.add(new GenotypeBuilder(genotype).alleles(fixedAlleles).make());
        }
        return fixedGenotypes;
    }
}

