#!/bin/bash
## use #!/bin/bash -x for debugging

## Galaxy wrapper for MUMmer (nucmer/promer)
## Alex Bossers, CVI of Wageningen UR, NL
## alex_dot_bossers_at_wur_dot_nl
##
## Sep 2010
##
## Wrapper runs MUMmer nucmer/promer and additional args
## Calculates the comparison scores (delta and optional coords file)
## Generates the optional STATIC comparison mummerplot to png (from delta file)
##
## finally the script renames (optional) output files to outfiles expected by Galaxy
##
##
## INPUT args:
## nucmer_tool.sh $input_ref $input_query $out_delta $out_coords $out_png $logfile 
##                    @0          @1          @2          @3        @4       @5   
##                $algorithm $keep_delta $make_coords $keep_log $make_image $cmd_extra 
##                     @6        @7           @8          @9        @10         @11
##

# Function to send error messages.
log_err() { echo "$@" 1>&2; }
# path to where mummer suite is installed
# adjust this for your machine
# If mummer is available in system path, leave empty
# when using different path make sure the trailing slash is added.
# mum_path = /opt/Mummer23/Mummer/
mum_path=""
tmp_path="/tmp/mummertmp/"

if [ $num_path"$(which mummer)" == "" ] && [ "$num_path" == "" ]; then
	log_err "mummer is not available in system path and not declarated in mum_path. Please install mummer."
	exit 127 
fi

# since we have more than 9 arguments we need to shift the sections or use own array
args=("$@")
# to keep things readible assign vars
input_ref="${args[0]}"
input_query="${args[1]}"
out_delta="${args[2]}"
out_coords="${args[3]}"
out_png="${args[4]}"
logfile="${args[5]}"
algorithm="${args[6]}"
keep_delta="${args[7]}"
make_coords="${args[8]}"
keep_log="${args[9]}"
make_image="${args[10]}"
cmd_extra="${args[11]}"

[ -d $tmp_path ] || mkdir $tmp_path
cd $tmp_path

# enable/disable the STDOUT log file
if [ "$keep_log" == "yes" ]; then
	logfile_c="2>$logfile"
	logfile_a="2>>$logfile"
else
	#dump to dev/null
	logfile_c="2>&-"
	logfile_a="2>&-"
fi

# extra mummer cmd line options

## generate coords file on the fly?
if [ "$make_coords" == "yes" ]; then
	options=" --coords"
fi
## extra cmd line args to be concatenated in options? We need to prevent extra spaces!
if [ "$cmd_extra" != "" ]; then
	if [ "$options" == "" ]; then
		options=" $cmd_extra"
	else
		options="$options $cmd_extra"
	fi
fi

# run nucmer/promer
# May only run Promer and Nucmer
echo $algorithm
if [[ $algorithm =~ ...mer$ ]]; then
	eval "$mum_path$algorithm$options $input_ref $input_query $logfile_c"
else 
	log_err 'ERROR, algorithm does not conform to ...mer'
	exit 1
fi


## generate large png if option make_image = yes
## suppress error from mummerplot since some is deprecated but not a real error
## error can be easily avoided by modifying the source of mummerplot... just in case
## however we need to check if a valid png was generated. This is not the case is alignment is none
## 1 is stderr and 2 stdout. redirect to dev/null
if [ "${make_image}" == "yes" ]; then
	eval "$mum_path mummerplot --large --png out.delta 1>&- $logfile_a"
	if [ -f "out.png" ]; then
		mv out.png $out_png
		#cleanup temp gnuplot file
		rm out.gp
	else
		log_err "not exist the req png file!"
		exit 1
	fi
	
	## clean up remaining files
	rm out.fplot
	rm out.rplot

fi

# keep/rename or delete delta file
if [ "$keep_delta" == "yes" ]; then
	mv out.delta "$out_delta" 
else
	rm out.delta
fi

# keep/rename coords file if it was created
if [ "$make_coords" == "yes" ]; then
	mv out.coords "$out_coords"
fi
# end script
exit 0